<?php
/**
 * Exit if accessed directly
 */
defined( 'ABSPATH' ) || exit( 'دسترسی غیر مجاز!' );

/**
 * Get image width and height by given url
 *
 * @param string $url
 *
 * @return array $info
 */
function wip_get_image_dim( $url ) {
	if ( ! $url ) {
		return array( 'h' => 0, 'w' => 0 );
	}
	list( $width, $height ) = getimagesize( $url );
	$info = array( 'h' => $height, 'w' => $width );

	return $info;
}

/**
 * Check all array value is numeric
 *
 * @param $array
 *
 * @return bool
 */
function wip_is_numeric_array( $array ) {
	foreach ( $array as $a => $b ) {
		if ( ! is_numeric( $b ) ) {
			return false;
		}
	}

	return true;
}

/**
 * Convert CSS hex color to rgba
 *
 * @param $color
 * @param bool $opacity
 *
 * @return string
 * https://gist.github.com/colourstheme/d992abc081df381ce656
 */
function wip_convert_hex2rgba_srezasalehi( $color, $opacity = false ) {
	$default = 'rgb(0,0,0)';

	if ( empty( $color ) ) {
		return $default;
	}

	if ( $color[0] == '#' ) {
		$color = substr( $color, 1 );
	}

	if ( strlen( $color ) == 6 ) {
		$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
	} elseif ( strlen( $color ) == 3 ) {
		$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
	} else {
		return $default;
	}

	$rgb = array_map( 'hexdec', $hex );

	if ( $opacity ) {
		if ( abs( $opacity ) > 1 ) {
			$opacity = 1.0;
		}

		$output = 'rgba(' . implode( ",", $rgb ) . ',' . $opacity . ')';
	} else {
		$output = 'rgb(' . implode( ",", $rgb ) . ')';
	}

	return $output;
}

/**
 * Check current page is in Dokan dashboard page
 * @return bool
 */
function wip_is_dokan_dashboard_page() {
	if ( in_array( 'dokan-lite/dokan.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
		$dokan_pages = get_option( 'dokan_pages' );
		if ( is_array( $dokan_pages ) ) {
			$dashboard = array_key_exists( 'dashboard', $dokan_pages ) ? $dokan_pages['dashboard'] : null;
			if ( $dashboard == get_the_ID() ) {
				return true;
			}
		}
	}

	return false;
}

/**
 * Retrieve refunded product form order
 *
 * @param $order_id
 *
 * @return array | null
 *
 * @since 8.0.0
 *
 * https://docs.woocommerce.com/wc-apidocs/class-WC_Order_Item_Product.html
 */
function wip_get_refunded_products( $order_id ) {
	if ( ! $order_id ) {
		return null;
	}

	$refund_items = get_posts( array(
		'post_type'   => 'shop_order_refund',
		'post_parent' => $order_id,
		'post_status' => 'any',
	) );

	if ( ! $refund_items || ! is_array( $refund_items ) ) {
		return null;
	}

	$products = array();

	foreach ( $refund_items as $refund_id ) {
		$refund = new WC_Order_Refund( $refund_id );
		$items  = $refund->get_items();

		foreach ( $items as $item ) {
			$product = $item->get_product();

			$type         = $product->get_type();
			$product_id   = $item->get_product_id();
			$variation_id = $item->get_variation_id();
			$subtotal     = $item->get_subtotal();
			$quantity     = $item->get_quantity();

			$p = array();

			if ( $type == 'variation' ) {
				$p['id'] = $variation_id;
			} elseif ( $type == 'simple' ) {
				$p['id'] = $product_id;
			}

			$p['subtotal'] = $subtotal;
			$p['quantity'] = $quantity;

			$products[] = $p;

		}
	}

	return $products;
}

/**
 * Gets the main product image
 *
 * @param $product
 * @param string $size (default: 'shop_thumbnail')
 *
 * @param array $attr
 *
 * @return string
 */
function wip_get_product_image( $product, $size = 'shop_thumbnail', $attr = array() ) {
	$product_id = $product->get_id();
	if ( $product->get_type() == 'variation' && has_post_thumbnail( $product_id ) ) {
		$image = get_the_post_thumbnail( $product_id, $size, $attr );
	} elseif ( has_post_thumbnail( $product_id ) ) {
		$image = get_the_post_thumbnail( $product_id, $size, $attr );
	} elseif ( ( $parent_id = wp_get_post_parent_id( $product_id ) ) && has_post_thumbnail( $parent_id ) ) {
		$image = get_the_post_thumbnail( $parent_id, $size, $attr );
	} else {
		$image = wc_placeholder_img( $size );
	}

	return $image;
}

/**
 * Get all orders
 */
function wip_get_orders() {
	$args = array(
		'limit'   => - 1,
		'orderby' => 'date',
		'order'   => 'DESC',
		'return'  => 'ids',
	);

	if ( isset( $_GET['state_filter'] ) && ! empty( $_GET['state_filter'] ) ) {
		// TODO billing or shipping
		$args['billing_state'] = $_GET['state_filter'];
	}

	if ( isset( $_GET['status_filter'] ) && ! empty( $_GET['status_filter'] ) ) {
		$args['status'] = $_GET['status_filter'];
	}

	if ( isset( $_GET['date_start'] ) && ! empty( $_GET['date_start'] )
	     && isset( $_GET['date_end'] ) && ! empty( $_GET['date_end'] ) ) {
		$date_start = $_GET['date_start'];
		$date_end   = $_GET['date_end'];

		list( $date_start_y, $date_start_m, $date_start_d ) = explode( '-', $date_start );
		list( $date_end_y, $date_end_m, $date_end_d ) = explode( '-', $date_end );

		$date_start_g = wip_jalali_to_gregorian( $date_start_y, $date_start_m, $date_start_d );
		$date_start_g = implode( $date_start_g, '-' );
		$date_end_g   = wip_jalali_to_gregorian( $date_end_y, $date_end_m, $date_end_d );
		$date_end_g   = implode( $date_end_g, '-' );

		date_default_timezone_set( "UTC" );

		$date_start_g = strtotime( $date_start_g );
		$date_end_g   = strtotime( $date_end_g );


		$args['date_created'] = $date_start_g . '...' . $date_end_g;
	}

	$query = new WC_Order_Query( $args );

	$orders = $query->get_orders();

	return $orders;
}

/**
 * Get all states of specific country
 *
 * @param $country
 *
 * @return mixed
 */
function wip_get_states( $country ) {
	$countries_obj = new WC_Countries();

	$country_states_array = $countries_obj->get_states();

	return $country_states_array[ $country ];
}

/**
 * Get state name by its code
 *
 * @param $country
 * @param $state
 *
 * @return mixed
 */
function wip_get_state_name( $country, $state ) {
	$state_name = wip_get_states( $country ) [ $state ];

	return $state_name;
}

/**
 * Get HTML output of states
 */
function wip_html_states() {
	?>
    <select name="state_filter">
        <option value="">فیلتر براساس استان</option>
		<?php
		$states = wip_get_states( 'IR' );
		if ( $states ) {
			foreach ( $states as $key => $state ) {
				$state_filter = null;
				if ( isset( $_GET['state_filter'] ) && ! empty( $_GET['state_filter'] ) ) {
					$state_filter = $_GET['state_filter'];
				}
				?>
                <option value="<?php echo $key; ?>" <?php selected( $state_filter, $key ); ?>><?php echo $state; ?></option>
				<?php
			}
		}
		?>
    </select>
	<?php
}

/**
 * Get HTML output of order statuses
 */
function wip_html_order_statuses() {
	?>
    <select name="status_filter">
        <option value="">فیلتر براساس وضعیت</option>
		<?php
		$statuses = wc_get_order_statuses();
		if ( $statuses ) {
			foreach ( $statuses as $key => $status ) {
				$status_filter = null;
				if ( isset( $_GET['status_filter'] ) && ! empty( $_GET['status_filter'] ) ) {
					$status_filter = $_GET['status_filter'];
				}
				?>
                <option value="<?php echo $key; ?>" <?php selected( $status_filter, $key ); ?>><?php echo $status; ?></option>
				<?php
			}
		}
		?>
    </select>
	<?php
}

/**
 * Convert jalali date to gregorian date
 * @url https://jdf.scr.ir/
 *
 * @param $jy
 * @param $jm
 * @param $jd
 * @param string $mod
 *
 * @return array|string
 */
function wip_jalali_to_gregorian( $jy, $jm, $jd, $mod = '' ) {
	if ( $jy > 979 ) {
		$gy = 1600;
		$jy -= 979;
	} else {
		$gy = 621;
	}
	$days = ( 365 * $jy ) + ( ( (int) ( $jy / 33 ) ) * 8 ) + ( (int) ( ( ( $jy % 33 ) + 3 ) / 4 ) ) + 78 + $jd + ( ( $jm < 7 ) ? ( $jm - 1 ) * 31 : ( ( $jm - 7 ) * 30 ) + 186 );
	$gy   += 400 * ( (int) ( $days / 146097 ) );
	$days %= 146097;
	if ( $days > 36524 ) {
		$gy   += 100 * ( (int) ( -- $days / 36524 ) );
		$days %= 36524;
		if ( $days >= 365 ) {
			$days ++;
		}
	}
	$gy   += 4 * ( (int) ( $days / 1461 ) );
	$days %= 1461;
	if ( $days > 365 ) {
		$gy   += (int) ( ( $days - 1 ) / 365 );
		$days = ( $days - 1 ) % 365;
	}
	$gd = $days + 1;
	foreach (
		array(
			0,
			31,
			( ( $gy % 4 == 0 and $gy % 100 != 0 ) or ( $gy % 400 == 0 ) ) ? 29 : 28,
			31,
			30,
			31,
			30,
			31,
			31,
			30,
			31,
			30,
			31
		) as $gm => $v
	) {
		if ( $gd <= $v ) {
			break;
		}
		$gd -= $v;
	}

	return ( $mod == '' ) ? array( $gy, $gm, $gd ) : $gy . $mod . $gm . $mod . $gd;
}

/**
 * Retrieve orders list print URL
 *
 * @return string
 */
function wip_get_orders_print_url() {
	$query = $_SERVER['QUERY_STRING'];

	$query = str_replace( 'page=wip-orders', '', $query );

	$query = '?wip-print-orders=1&_wip_nonce=' . wp_create_nonce( '_wip_nonce' ) . $query;

	$url = admin_url() . $query;

	return $url;
}

/**
 * Generate variation product full name
 *
 * @param $product
 *
 * @return null|string
 */
function wip_get_variation_name( $product ) {
	if ( $product->get_type() == 'variation' ) {
		$name = $product->get_title();

		$attributes = $product->get_attributes();
		foreach ( $attributes as $key => $value ) {
			$name .= ' - ' . $product->get_attribute( $key );
		}

		return $name;
	}

	return null;
}